/*
 * Decompiled with CFR 0.152.
 */
package pl.skidam.automodpack_loader_core.utils;

import java.io.BufferedInputStream;
import java.io.FileOutputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.zip.GZIPInputStream;
import pl.skidam.automodpack_core.GlobalVariables;
import pl.skidam.automodpack_core.utils.CustomFileUtils;
import pl.skidam.automodpack_core.utils.CustomThreadFactoryBuilder;

public class DownloadManager {
    private static final int MAX_DOWNLOADS_IN_PROGRESS = 5;
    private static final int MAX_DOWNLOAD_ATTEMPTS = 2;
    private static final int BUFFER_SIZE = 131072;
    private final ExecutorService DOWNLOAD_EXECUTOR = Executors.newFixedThreadPool(5, new CustomThreadFactoryBuilder().setNameFormat("AutoModpackDownload-%d").build());
    private final Map<HashAndPath, QueuedDownload> queuedDownloads = new ConcurrentHashMap<HashAndPath, QueuedDownload>();
    public final Map<HashAndPath, DownloadData> downloadsInProgress = new ConcurrentHashMap<HashAndPath, DownloadData>();
    private long bytesDownloaded = 0L;
    private long bytesToDownload = 0L;
    private int addedToQueue = 0;
    private int downloaded = 0;
    private final Semaphore semaphore = new Semaphore(0);

    public DownloadManager() {
    }

    public DownloadManager(long bytesToDownload) {
        this.bytesToDownload = bytesToDownload;
    }

    public void download(Path file, String sha1, Urls urls, Runnable successCallback, Runnable failureCallback) {
        HashAndPath hashAndPath = new HashAndPath(sha1, file);
        if (this.queuedDownloads.containsKey(hashAndPath)) {
            return;
        }
        this.queuedDownloads.put(hashAndPath, new QueuedDownload(file, urls, 0, successCallback, failureCallback));
        ++this.addedToQueue;
        this.downloadNext();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void downloadTask(HashAndPath hashAndPath, QueuedDownload queuedDownload) {
        GlobalVariables.LOGGER.info("Downloading {} - {}", (Object)queuedDownload.file.getFileName(), (Object)queuedDownload.urls.toString());
        int numberOfIndexes = queuedDownload.urls.numberOfUrls - 1;
        int urlIndex = Math.min(queuedDownload.attempts / 2, numberOfIndexes);
        String url = queuedDownload.urls.URLs.get((int)(numberOfIndexes - urlIndex)).url;
        boolean interrupted = false;
        try {
            this.downloadFile(url, hashAndPath, queuedDownload);
        }
        catch (InterruptedException e) {
            CustomFileUtils.forceDelete(queuedDownload.file);
            interrupted = true;
        }
        catch (SocketTimeoutException e) {
            CustomFileUtils.forceDelete(queuedDownload.file);
            GlobalVariables.LOGGER.warn("Timeout - {} - {}", (Object)queuedDownload.file, (Object)e);
        }
        catch (Exception e) {
            CustomFileUtils.forceDelete(queuedDownload.file);
            GlobalVariables.LOGGER.warn("Error while downloading file - {} - {}", (Object)queuedDownload.file, (Object)e);
        }
        finally {
            Object hash;
            Map<HashAndPath, DownloadData> e = this.downloadsInProgress;
            synchronized (e) {
                this.downloadsInProgress.remove(hashAndPath);
            }
            boolean failed = true;
            if (Files.exists(queuedDownload.file, new LinkOption[0])) {
                hash = CustomFileUtils.getHash(queuedDownload.file, "SHA-1").orElse(null);
                if (!Objects.equals(hash, hashAndPath.hash)) {
                    this.bytesDownloaded -= queuedDownload.file.toFile().length();
                } else {
                    failed = false;
                    ++this.downloaded;
                    GlobalVariables.LOGGER.info("Successfully downloaded {} from {}", (Object)queuedDownload.file.getFileName(), (Object)url);
                    queuedDownload.successCallback.run();
                    this.semaphore.release();
                }
            }
            if (failed) {
                CustomFileUtils.forceDelete(queuedDownload.file);
                if (interrupted) {
                    return;
                }
                if (queuedDownload.attempts < queuedDownload.urls.numberOfUrls * 2) {
                    GlobalVariables.LOGGER.warn("Download of {} failed, retrying!", (Object)queuedDownload.file.getFileName());
                    ++queuedDownload.attempts;
                    hash = this.queuedDownloads;
                    synchronized (hash) {
                        this.queuedDownloads.put(hashAndPath, queuedDownload);
                    }
                } else {
                    GlobalVariables.LOGGER.error("Download of {} failed!", (Object)queuedDownload.file.getFileName());
                    queuedDownload.failureCallback.run();
                    this.semaphore.release();
                }
            }
            this.downloadNext();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void downloadNext() {
        if (this.downloadsInProgress.size() < 5 && !this.queuedDownloads.isEmpty()) {
            HashAndPath hashAndPath = (HashAndPath)this.queuedDownloads.keySet().stream().findFirst().get();
            QueuedDownload queuedDownload = this.queuedDownloads.remove(hashAndPath);
            if (queuedDownload == null) {
                return;
            }
            CompletableFuture<Void> future = CompletableFuture.runAsync(() -> this.downloadTask(hashAndPath, queuedDownload), this.DOWNLOAD_EXECUTOR);
            Map<HashAndPath, DownloadData> map = this.downloadsInProgress;
            synchronized (map) {
                this.downloadsInProgress.put(hashAndPath, new DownloadData(future, queuedDownload.file));
            }
        }
    }

    private void downloadFile(String urlString, HashAndPath hashAndPath, QueuedDownload queuedDownload) throws IOException, InterruptedException {
        Path outFile = queuedDownload.file;
        if (Files.exists(outFile, new LinkOption[0])) {
            if (Objects.equals(hashAndPath.hash, CustomFileUtils.getHash(outFile, "SHA-1").orElse(null))) {
                return;
            }
            CustomFileUtils.forceDelete(outFile);
        }
        if (outFile.getParent() != null) {
            Files.createDirectories(outFile.getParent(), new FileAttribute[0]);
        }
        Files.createFile(outFile, new FileAttribute[0]);
        URL url = new URL(urlString);
        URLConnection connection = url.openConnection();
        connection.addRequestProperty("Accept-Encoding", "gzip");
        connection.addRequestProperty("User-Agent", "github/skidamek/automodpack/" + GlobalVariables.AM_VERSION);
        connection.setConnectTimeout(10000);
        connection.setReadTimeout(10000);
        try (FileOutputStream outputStream = new FileOutputStream(outFile.toFile());
             BufferedInputStream rawInputStream = new BufferedInputStream(connection.getInputStream(), 131072);
             FilterInputStream inputStream = "gzip".equals(connection.getHeaderField("Content-Encoding")) ? new GZIPInputStream(rawInputStream) : rawInputStream;){
            int bytesRead;
            byte[] buffer = new byte[131072];
            while ((bytesRead = ((InputStream)inputStream).read(buffer)) != -1) {
                this.bytesDownloaded += (long)bytesRead;
                ((OutputStream)outputStream).write(buffer, 0, bytesRead);
                if (!Thread.currentThread().isInterrupted()) continue;
                throw new InterruptedException("Download got cancelled");
            }
        }
    }

    public void joinAll() throws InterruptedException {
        this.semaphore.acquire(this.addedToQueue);
        if (this.DOWNLOAD_EXECUTOR.isShutdown()) {
            throw new InterruptedException();
        }
        this.semaphore.release(this.addedToQueue);
    }

    public long getTotalDownloadSpeed() {
        long totalBytesDownloaded = this.downloadsInProgress.values().stream().mapToLong(data -> data.file.toFile().length()).sum();
        long totalDownloadTimeInSeconds = this.downloadsInProgress.values().stream().mapToLong(data -> Duration.between(data.startTime, Instant.now()).getSeconds()).sum();
        return totalDownloadTimeInSeconds == 0L ? 0L : totalBytesDownloaded / totalDownloadTimeInSeconds;
    }

    public String getTotalDownloadSpeedInReadableFormat(long totalDownloadSpeed) {
        if (totalDownloadSpeed == 0L) {
            return "-1";
        }
        return this.addUnitsPerSecond(totalDownloadSpeed);
    }

    public String getTotalETA(long totalDownloadSpeed) {
        long totalBytesRemaining = this.bytesToDownload - this.bytesDownloaded;
        return totalDownloadSpeed == 0L ? "0" : totalBytesRemaining / totalDownloadSpeed + "s";
    }

    public String addUnitsPerSecond(long size) {
        int unitIndex;
        String[] units = new String[]{"B", "KB", "MB", "GB", "TB"};
        for (unitIndex = 0; size > 1024L && unitIndex < units.length - 1; size /= 1024L, ++unitIndex) {
        }
        return size + units[unitIndex] + "/s";
    }

    public float getTotalPercentageOfFileSizeDownloaded() {
        return (float)this.bytesDownloaded / (float)this.bytesToDownload * 100.0f;
    }

    public String getStage() {
        return this.downloaded + "/" + this.addedToQueue;
    }

    public boolean isClosed() {
        return this.DOWNLOAD_EXECUTOR.isShutdown();
    }

    public void cancelAllAndShutdown() {
        this.queuedDownloads.clear();
        this.downloadsInProgress.forEach((url, downloadData) -> {
            downloadData.future.cancel(true);
            CustomFileUtils.forceDelete(downloadData.file);
        });
        this.semaphore.release(this.addedToQueue);
        this.downloadsInProgress.clear();
        this.downloaded = 0;
        this.addedToQueue = 0;
        this.DOWNLOAD_EXECUTOR.shutdownNow();
        try {
            if (!this.DOWNLOAD_EXECUTOR.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.DOWNLOAD_EXECUTOR.shutdownNow();
                if (!this.DOWNLOAD_EXECUTOR.awaitTermination(3L, TimeUnit.SECONDS)) {
                    GlobalVariables.LOGGER.error("DOWNLOAD EXECUTOR did not terminate");
                }
            }
        }
        catch (InterruptedException e) {
            this.DOWNLOAD_EXECUTOR.shutdownNow();
        }
    }

    public record HashAndPath(String hash, Path path) {
    }

    public static class QueuedDownload {
        private final Path file;
        private final Urls urls;
        private int attempts;
        private final Runnable successCallback;
        private final Runnable failureCallback;

        public QueuedDownload(Path file, Urls urls, int attempts, Runnable successCallback, Runnable failureCallback) {
            this.file = file;
            this.urls = urls;
            this.attempts = attempts;
            this.successCallback = successCallback;
            this.failureCallback = failureCallback;
        }
    }

    public static class Urls {
        private final List<Url> URLs = new ArrayList<Url>(3);
        private int numberOfUrls;

        public Urls addUrl(Url url) {
            this.URLs.add(url);
            this.numberOfUrls = this.URLs.size();
            return this;
        }

        public Urls addAllUrls(List<Url> urls) {
            this.URLs.addAll(urls);
            this.numberOfUrls = this.URLs.size();
            return this;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            this.URLs.forEach(url -> sb.append(url.url).append(", "));
            sb.delete(sb.length() - 2, sb.length());
            String str = sb.toString();
            return "[" + str + "]";
        }
    }

    public static class Url {
        private final Map<String, String> headers = new HashMap<String, String>();
        private String url;

        public Url getUrl(String url) {
            this.url = url;
            return this;
        }

        public List<Url> getUrls(List<String> urls) {
            ArrayList<Url> urlList = new ArrayList<Url>();
            urls.forEach(url -> urlList.add(new Url().getUrl((String)url)));
            return urlList;
        }

        public void addHeader(String headerName, String header) {
            this.headers.put(headerName, header);
        }
    }

    public static class DownloadData {
        public CompletableFuture<Void> future;
        public Path file;
        public final Instant startTime = Instant.now();

        DownloadData(CompletableFuture<Void> future, Path file) {
            this.future = future;
            this.file = file;
        }

        public String getFileName() {
            return this.file.getFileName().toString();
        }
    }
}

